//
//  ASReachability.m
//  CRMNetDetect
//
//  Created by 坚鹏 on 2024/10/14.
//

#import "ASReachability.h"
#import <SystemConfiguration/SystemConfiguration.h>
#import <netinet/in.h>

@interface ASReachability()

@property(nonatomic,strong,nullable)ASCellularService * celluarService;
@property (nonatomic) SCNetworkReachabilityRef reachabilityRef;
@end
@implementation ASReachability

-(instancetype)initWithDelegate:(id<ASReachabilityDelegate>)delegate
{
    if(self = [super init]){
        _delegate = delegate;
    }return self;
}

-(void)startsObserving
{
    if(_reachabilityRef){
        return;
    }
    self.reachabilityRef = [self _createHostReachabilityRef]?:[self _createInternetReachabilityRef];
    if(!_reachabilityRef){
        NSLog(@"Could not create reachabilityRef");
        return;
    }
    // 初始化时检查网络状态
    [self checkCurrentNetworkStatus];
    
    SCNetworkReachabilityContext context = {0, (__bridge void *)(self), NULL, NULL, NULL};
    SCNetworkReachabilitySetCallback(self.reachabilityRef, ReachabilityCallback, &context);
    SCNetworkReachabilityScheduleWithRunLoop(self.reachabilityRef, CFRunLoopGetCurrent(), kCFRunLoopCommonModes);
}
-(void)stopObserving
{
    if (_reachabilityRef) {
        SCNetworkReachabilityUnscheduleFromRunLoop(_reachabilityRef, CFRunLoopGetCurrent(), kCFRunLoopCommonModes);
        CFRelease(_reachabilityRef);
        _reachabilityRef = NULL;
    }
}
// 网络状态变化回调函数
static void ReachabilityCallback(SCNetworkReachabilityRef target, SCNetworkReachabilityFlags flags, void *info) {
    ASReachability *monitor = (__bridge ASReachability *)info;
    [monitor handleNetworkChangeWithFlags:flags];
}
- (void)checkCurrentNetworkStatus {
    SCNetworkReachabilityFlags flags;
    if (SCNetworkReachabilityGetFlags(self.reachabilityRef, &flags)) {
        [self handleNetworkChangeWithFlags:flags];
    } else {
        NSLog(@"Failed to retrieve network status");
    }
}
// 处理网络状态变化
- (void)handleNetworkChangeWithFlags:(SCNetworkReachabilityFlags)flags {
    BOOL isReachable = (flags & kSCNetworkReachabilityFlagsReachable);
    BOOL needsConnection = (flags & kSCNetworkReachabilityFlagsConnectionRequired);
    if (isReachable && !needsConnection) {
        NSLog(@"Network is reachable");
        // 进一步检查网络类型
        [self detectConnectionTypeWithFlags:flags];
    } else {
        BOOL isTransient = (flags & kSCNetworkReachabilityFlagsTransientConnection);
        BOOL interventionRequired = (flags & kSCNetworkReachabilityFlagsInterventionRequired);
        // 网络不可达处理
        if (isReachable && interventionRequired) {
            NSLog(@"Network requires user intervention (e.g., Wi-Fi login)");
        } else if (isReachable && isTransient) {
            NSLog(@"Network connection is transient (e.g., dial-up or other temporary connection)");
        } else {
            NSLog(@"Network is not reachable");
        }
        _status = ASReachabilityStatusNotReachable;
        [self.delegate reachabilityBecomesNotReachable:self];
    }
}

// 检测当前的网络连接类型（Wi-Fi 或 蜂窝）
- (void)detectConnectionTypeWithFlags:(SCNetworkReachabilityFlags)flags {
    if ((flags & kSCNetworkReachabilityFlagsIsWWAN) != 0) {
        // 通过蜂窝网络连接
        
        if(!_celluarService){_celluarService = [ASCellularService new];}
        ASCellularType type = _celluarService.accessType;
        NSLog(@"Connected via Cellular of %ld",type);
        _status = ASReachabilityStatusWifi;
        [self.delegate reachabilityChangedToWWAN:self type:type signalStrength:[_celluarService signalStrengthEvaluatedForType:type]];
    } else {
        // 通过Wi-Fi或其他网络连接
        NSLog(@"Connected via Wi-Fi");
        _status = ASReachabilityStatusWWAN;
        [self.delegate reachabilityChangedToWiFi:self];
    }
}

-(void)dealloc
{
    [self stopObserving];
}
#pragma mark - Privates
- (SCNetworkReachabilityRef _Nullable)_createHostReachabilityRef
{
    return _host ? SCNetworkReachabilityCreateWithName(NULL, _host.UTF8String) : nil;
}
- (SCNetworkReachabilityRef _Nullable)_createInternetReachabilityRef{
    struct sockaddr_in6 ipv6Address;
    memset(&ipv6Address, 0, sizeof(ipv6Address));
    ipv6Address.sin6_len = sizeof(ipv6Address);
    ipv6Address.sin6_family = AF_INET6;
    SCNetworkReachabilityRef  ref = SCNetworkReachabilityCreateWithAddress(NULL, (struct sockaddr *)&ipv6Address);
    if(ref){return ref;}//优先使用ipv6地址
    struct sockaddr_in ipv4Address;
    memset(&ipv4Address, 0, sizeof(ipv4Address));
    ipv4Address.sin_len = sizeof(ipv4Address);
    ipv4Address.sin_family = AF_INET;
    ref = SCNetworkReachabilityCreateWithAddress(NULL, (struct sockaddr *)&ipv4Address);
    return ref;
}
@end
