//
//  ASPingService.m
//  CRMNetDetect
//
//  Created by 坚鹏 on 2024/10/15.
//

#import "ASPingService.h"
#import <CRMNetDetect/ASSimplePing.h>
#import <sys/time.h>

static uint16_t KASPingNotStartedSeq = -1;
@interface ASPingService()<ASSimplePingDelegate>

@end
@implementation ASPingService
{
    ASSimplePing *_samplePing;
    NSTimeInterval _startPingTime;
    uint16_t _pingingSeq;
}
-(instancetype)initWithDelegate:(id<ASPingServiceDelegate>)delegate{
    if (self = [super init]) {
        _delegate = delegate;
        _pingCount = 1;
        _timeoutSeconds = 3;
        _pingingSeq = KASPingNotStartedSeq;
    }return self;
}
-(void)starts
{
    if(_isPinging){
        if([self.delegate respondsToSelector:@selector(pingService:failedToStartWithError:)]){
            [self.delegate pingService:self failedToStartWithError:[self _createErrorWithMsg:@"Pinging Aready!"]];
        }
        return;
    }
    NSAssert(_targetHost, @"target host could not be nil.");
    NSAssert(_pingCount >=1, @"Ping count could not smaller than 1");
    NSAssert(_timeoutSeconds >=1, @"Ping timeout could not smaller than 1");
    _isPinging = true;
    _samplePing = [[ASSimplePing alloc] initWithHostName:_targetHost];
    _samplePing.delegate = self;
    _samplePing.addressStyle = _usingICMPv6?ASSimplePingAddressStyleICMPv6:ASSimplePingAddressStyleICMPv4;
    if([self.delegate respondsToSelector:@selector(pingingServiceDidStart:)]){
        [self.delegate pingingServiceDidStart:self];
    }
    NSLog(@"PING %@:",_targetHost);
    [_samplePing start];
}
-(void)stops
{
    if(!_samplePing){
        return;
    }
    NSLog(@"stops ping service");
    [_samplePing stop];
    _samplePing = nil;
    _pingingSeq = KASPingNotStartedSeq;
    _isPinging = false;
    [NSObject cancelPreviousPerformRequestsWithTarget:self];
}
- (NSError*)_createErrorWithMsg:(NSString*)msg
{
    return [NSError errorWithDomain:@"ASPingServiceErrorDomain" code:0 userInfo:@{NSLocalizedDescriptionKey:msg}];
}
- (void)_sendPingOnce
{
    [NSObject cancelPreviousPerformRequestsWithTarget:self selector:@selector(_gotManulyTimeoutNotification) object:nil];
    _pingingSeq ++;
    if(_pingingSeq >= _pingCount){
        [self stops];
        if([self.delegate respondsToSelector:@selector(pingService:stopWithError:)]){
            [self.delegate pingService:self stopWithError:nil];
        }
        return;
    }
    [self performSelector:@selector(_gotManulyTimeoutNotification) withObject:nil afterDelay:self.timeoutSeconds];
    _startPingTime = [self _getCurrentMicroSeconds];
    [_samplePing sendPingWithData:nil];
    NSLog(@"did send next ping and current count is %hu",_pingingSeq);
}
- (void)_gotManulyTimeoutNotification
{
    NSLog(@"Request timeout for icmp_seq %hu",_pingingSeq);
    //当次ping超时
    if([self.delegate respondsToSelector:@selector(pingService:pingTimeout:)]){
        [self.delegate pingService:self pingTimeout:_pingingSeq];
    }
    //开启下次ping
    [self _sendPingOnce];
}
- (NSTimeInterval)_getCurrentMicroSeconds
{
    struct timeval time;
    gettimeofday(&time,NULL);
    return time.tv_usec;
}
- (NSTimeInterval)_getPingTimePastMicroSeconds
{
    NSTimeInterval now = [self _getCurrentMicroSeconds];
    if(now < _startPingTime){
        return (1000000 + now - _startPingTime)/1000;
    }
    return (now - _startPingTime)/1000;
}
#pragma mark - ASSimplePingDelegate
- (void)simplePing:(ASSimplePing *)pinger didStartWithAddress:(NSData *)address
{
    [self _sendPingOnce];
}
- (void)simplePing:(ASSimplePing *)pinger didFailWithError:(NSError *)error
{
    BOOL isTimeoutCase = ([error.domain isEqualToString:NSPOSIXErrorDomain]&&error.code ==ETIMEDOUT);
    if(isTimeoutCase){
        [self _gotManulyTimeoutNotification];//按照手动超时的方式处理.
    }else{
        [self stops];
        if([self.delegate respondsToSelector:@selector(pingService:stopWithError:)]){
            [self.delegate pingService:self stopWithError:nil];
        }
    }
}
- (void)simplePing:(ASSimplePing *)pinger didSendPacket:(NSData *)packet sequenceNumber:(uint16_t)sequenceNumber
{
//    NSLog(@"did send ping seq:%hu",sequenceNumber);
}
- (void)simplePing:(ASSimplePing *)pinger didFailToSendPacket:(NSData *)packet sequenceNumber:(uint16_t)sequenceNumber error:(NSError *)error
{
    NSLog(@"did failed to send ping");
    if([self.delegate respondsToSelector:@selector(pingService:tmporaryErrorHappens:seq:)]){
        [self.delegate pingService:self tmporaryErrorHappens:error seq:_pingingSeq];
    }
    [self _sendPingOnce];//resend again.
}
- (void)simplePing:(ASSimplePing *)pinger didReceivePingResponsePacket:(NSData *)packet sequenceNumber:(uint16_t)sequenceNumber
{
    if(sequenceNumber == _pingingSeq){
        NSTimeInterval duration = [self _getPingTimePastMicroSeconds];
        NSLog(@"%ld bytes from %@: icmp_seq=%hu time=%.3f ms",packet.length,_targetHost,sequenceNumber,duration);
        [self.delegate pingService:self pingingWithDuration:duration seq:sequenceNumber];
        [self _sendPingOnce];
    }else{
        NSLog(@"got outdate sequenceNumber:%hu",sequenceNumber);
    }
}
- (void)simplePing:(ASSimplePing *)pinger didReceiveUnexpectedPacket:(NSData *)packet
{
    NSLog(@"did failed to send ping");
    if([self.delegate respondsToSelector:@selector(pingService:tmporaryErrorHappens:seq:)]){
        [self.delegate pingService:self tmporaryErrorHappens:[self _createErrorWithMsg:@"ReceiveUnexpectedPacket"] seq:_pingingSeq];
    }
    [self _sendPingOnce];//ignore,send again.
}
@end
