//
//  ASIPService.m
//  ASNetwork
//
//  Created by 坚鹏 on 2024/10/15.
//

#import "ASInterfaceService.h"
#include <ifaddrs.h>
#include <arpa/inet.h>
#include <net/if.h>
@interface ASIPEntity()
- (instancetype)initWithInterfaceName:(NSString*)name ip:(NSString*)ip isIpv6:(BOOL)isIpv6;
@end
@implementation ASInterfaceService

-(void)refreshAvailableInterfaces
{
    struct ifaddrs *interfaces = NULL;
    struct ifaddrs *temp_addr = NULL;
    int successCode = 0;
    if(getifaddrs(&interfaces)!=successCode){
        return;
    }
    temp_addr = interfaces;
    NSMutableArray * newIPList = @[].mutableCopy;
    while (temp_addr != NULL) {
        BOOL isUsed = (temp_addr->ifa_flags & IFF_UP);
        BOOL isIpv4 = (temp_addr->ifa_addr->sa_family == AF_INET);
        BOOL isIpv6 = (temp_addr->ifa_addr->sa_family == AF_INET6);
        if(!isUsed || (!isIpv4 && !isIpv6)){
            temp_addr = temp_addr->ifa_next;
            continue;
        }//只处理启用的ipv4或ipv6接口.
        NSString *interfaceName = [NSString stringWithUTF8String:temp_addr->ifa_name];
        NSString *ipAddress = nil;
        if(isIpv4){
            ipAddress = [NSString stringWithUTF8String:inet_ntoa(((struct sockaddr_in *)temp_addr->ifa_addr)->sin_addr)];
        }else{
            char addrBuffer[INET6_ADDRSTRLEN];
            const struct sockaddr_in6 *sockaddr_ipv6 = (const struct sockaddr_in6 *)temp_addr->ifa_addr;
            inet_ntop(AF_INET6, &sockaddr_ipv6->sin6_addr, addrBuffer, sizeof(addrBuffer));
            ipAddress = [NSString stringWithUTF8String:addrBuffer];
        }
        temp_addr = temp_addr->ifa_next;
        [newIPList addObject:[[ASIPEntity alloc]initWithInterfaceName:interfaceName ip:ipAddress isIpv6:isIpv6]];
    }
    freeifaddrs(interfaces);
    _availableIPList = newIPList;
}
-(NSArray<NSString *> *)getWiFiIP
{
    NSMutableArray * list;
    for (ASIPEntity * ipEntity in _availableIPList) {
        if(ipEntity.type == ASIPTypeWiFi){
            if(!list){list = @[].mutableCopy;}
            [list addObject:ipEntity.value];
        }
    }
    return list;
}
-(NSArray<NSString *> *)getCelluarIP
{
    NSMutableArray * list;
    for (ASIPEntity * ipEntity in _availableIPList) {
        if(ipEntity.type == ASIPTypeCelluar){
            if(!list){list = @[].mutableCopy;}
            [list addObject:ipEntity.value];
        }
    }
    return list;
}
-(BOOL)usingVPN
{
    for (ASIPEntity * ipEntity in _availableIPList) {
        if(ipEntity.type == ASIPTypeVPN){return true;}
    }
    return false;
}
@end
@implementation ASIPEntity
{
    NSString*_interfaceName;
}
-(instancetype)initWithInterfaceName:(NSString *)name ip:(NSString *)ip isIpv6:(BOOL)isIpv6{
    if(self = [super init]){
        _interfaceName = name;
        _value = ip;
        if(isIpv6){_isIpv6 = true;}else{_isIpv4 = true;}
        _type = [self _convertTypeFromInterfaceName:name];
        _isPrivateIP = [self isPrivateIP:_value];
        _isLinkLocalIP = _isIpv6?[self isIPv6LinkLocalAddress:ip]:[ip hasPrefix:@"169.254"];
    }return self;
}
- (ASIPType)_convertTypeFromInterfaceName:(NSString*)name
{
    if([name isEqualToString:@"en0"]){return ASIPTypeWiFi;}
    else if ([name isEqualToString:@"pdp_ip0"]){return ASIPTypeCelluar;}
    else if ([name hasPrefix:@"utun"]){return ASIPTypeVPN;}
    else if ([name hasPrefix:@"en"]){return ASIPTypeEthernet;}
    else if ([name hasPrefix:@"lo0"]){return ASIPTypeLo0;}
    else if ([name hasPrefix:@"bridge0"]){return ASIPTypeBridge0;}
    else{return ASIPTypeUnknown;}
}
-(BOOL)isPrivateIP:(NSString*)value
{
    if(_isIpv6){
        return [self isIPv6PrivateAddress:value];
    }
    //判断ipv4
    NSArray *privateRanges = @[
           @"10.",
           @"172.16.", @"172.17.", @"172.18.", @"172.19.", @"172.20.",
           @"172.21.", @"172.22.", @"172.23.", @"172.24.", @"172.25.",
           @"172.26.", @"172.27.", @"172.28.", @"172.29.", @"172.30.",
           @"172.31.",
           @"192.168."
       ];
    return [privateRanges containsObject:value];
}
// 判断 IPv6 地址是否为 Link-Local 地址
- (BOOL)isIPv6LinkLocalAddress:(NSString *)ipAddress {
    return [ipAddress hasPrefix:@"fe80"];
}

// 判断 IPv6 地址是否为私有 IP 地址（Unique Local Address, ULA）
- (BOOL)isIPv6PrivateAddress:(NSString *)ipAddress {
    return [ipAddress hasPrefix:@"fc00"] || [ipAddress hasPrefix:@"fd00"];
}
-(NSString *)description
{
    return [NSString stringWithFormat:@"IP:%@,interface:%@",_value,_interfaceName];
}
@end
